﻿uses
  Classes, Common, DataEntry, DataStorage, Regex, SysUtils;

const
  RPAD_VALUE = 25;
  CHAR_LENGTH = 80;

  // ------------------------------------------------------------------------------
  // Function: Right Pad v2
  // ------------------------------------------------------------------------------
function RPad(const AString: string; AChars: integer): string;
begin
  AChars := AChars - Length(AString);
  if AChars > 0 then
    Result := AString + StringOfChar(' ', AChars)
  else
    Result := AString;
end;

// ------------------------------------------------------------------------------
// Function: Create a PDF
// ------------------------------------------------------------------------------
procedure CreatePDF(aStringList: TStringList; FileSaveName_str: string; font_size: integer);
var
  CurrentCaseDir: string;
  SaveDir: string;
begin
  CurrentCaseDir := GetCurrentCaseDir;
  SaveDir := CurrentCaseDir + 'Reports\Run Time Reports\';
  if assigned(aStringList) and (aStringList.Count > 0) then
  begin
    if ForceDirectories(IncludeTrailingPathDelimiter(SaveDir)) and DirectoryExists(SaveDir) then
      GeneratePDF(SaveDir + FileSaveName_str, aStringList, font_size);
  end;
end;

// ------------------------------------------------------------------------------
// Function: Test Valid Regex
// ------------------------------------------------------------------------------
function IsValidRegEx(aStr: string): boolean;
var
  aRegEx: TRegEx;
begin
  Result := False;
  aRegEx := TRegEx.Create;
  try
    aRegEx.SearchTerm := aStr;
    Result := aRegEx.LastError = 0;
  finally
    aRegEx.free;
  end;
end;

// ------------------------------------------------------------------------------
// Function: Remove Last Part
// ------------------------------------------------------------------------------
function RemoveLastPart(var s: string): string;
begin
  repeat
    delete(s, Length(s), 1)
  until (s[Length(s)] = '\') or (s = '');
  Result := s;
end;

// ------------------------------------------------------------------------------
// On Start: Read the file from disk and create the regex string
// ------------------------------------------------------------------------------
procedure OnStartFilter(aDataStore: TDataStore; Var DoContinue: boolean; Var OutObj: string);
const
  KEYWORD_FILE_TXT = 'regex\regex_child_protection.txt';

var
  aKeywords_StringList: TStringList;
  FilterLogging_StringList: TStringList;
  CommentString: string;
  i: integer;
  OneBigRegex: string;
  FilterPath_str: string;
  Keyword_File_fullpath_str: string;
  trunc_str: string;

begin
  DoContinue := True;
  FilterLogging_StringList := TStringList.Create;
  aKeywords_StringList := TStringList.Create;
  aKeywords_StringList.Delimiter := '|';
  aKeywords_StringList.StrictDelimiter := True;

  try
    // FilterLogging_StringList.Add(Stringofchar('-', CHAR_LENGTH));
    // FilterLogging_StringList.Add(RPad('Filter:', RPAD_VALUE) + ExtractFileName(CmdLine.Path));
    FilterLogging_StringList.Add(StringOfChar('-', CHAR_LENGTH));

    // -------------------------------------------------------------------
    // Find the keyword file - The filter knows where it is...
    // -------------------------------------------------------------------
    FilterPath_str := CmdLine.Path;
    if RegexMatch(FilterPath_str, '\\filters', False) then
    begin

      // Remove the path back to the point where there is no filters folder
      while RegexMatch(FilterPath_str, '\\filters', False) do
        FilterPath_str := RemoveLastPart(FilterPath_str);

      // Step forward into the keywords folder
      if DirectoryExists(FilterPath_str) then
      begin
        Keyword_File_fullpath_str := FilterPath_str + 'keywords\' + KEYWORD_FILE_TXT;
      end;

    end;

    // -------------------------------------------------------------------
    // If the keyword file is found
    // -------------------------------------------------------------------
    if FileExists(Keyword_File_fullpath_str) then
    begin
      // Load from file
      aKeywords_StringList.LoadFromFile(Keyword_File_fullpath_str);

      // Remove blank lines from StringList
      for i := aKeywords_StringList.Count - 1 downto 0 do
      begin
        aKeywords_StringList[i] := TrimRight(aKeywords_StringList[i]);
        if Trim(aKeywords_StringList[i]) = '' then
          aKeywords_StringList.delete(i);
      end;

      // Remove comment lines from StringList
      for i := aKeywords_StringList.Count - 1 downto 0 do
      begin
        CommentString := copy(aKeywords_StringList(i), 1, 1);
        if CommentString = '#' then
          aKeywords_StringList.delete(i);
      end;
    end
    else
    begin
      FilterLogging_StringList.Add(RPad('Did not find:', RPAD_VALUE) + ExtractFileName(Keyword_File_fullpath_str));
    end;

    // -------------------------------------------------------------------
    // Create OnBigRegex and Test
    // -------------------------------------------------------------------
    OneBigRegex := aKeywords_StringList.DelimitedText;

    // -------------------------------------------------------------------
    // Logging
    // -------------------------------------------------------------------
    FilterLogging_StringList.Add(RPad('Keyword File:', RPAD_VALUE) + ExtractFileName(Keyword_File_fullpath_str));
    FilterLogging_StringList.Add(RPad('Keyword Count:', RPAD_VALUE) + IntToStr(aKeywords_StringList.Count));

    // Test the regex
    if IsValidRegEx(OneBigRegex) then
      FilterLogging_StringList.Add(RPad('RegEx Format Check:', RPAD_VALUE) + 'Valid')
    else
      FilterLogging_StringList.Add(RPad('RegEx Format Check:', RPAD_VALUE) + 'Invalid');
    // FilterLogging_StringList.Add('OneBigRegex: ' + OneBigRegex);

    FilterLogging_StringList.Add(StringOfChar('-', CHAR_LENGTH));
    FilterLogging_StringList.Add(' ');

    for i := 0 to aKeywords_StringList.Count - 1 do
    begin
      if Length(aKeywords_StringList(i)) > 45 then
      begin
        trunc_str := copy(aKeywords_StringList(i), 1, 45);
        FilterLogging_StringList.Add(RPad(IntToStr(i + 1) + ':', RPAD_VALUE) + trunc_str + '...');
      end
      else
        FilterLogging_StringList.Add(RPad(IntToStr(i + 1) + ':', RPAD_VALUE) + aKeywords_StringList(i));
    end;

    // -------------------------------------------------------------------
    // Create the Out Object
    // -------------------------------------------------------------------
    OutObj := OneBigRegex;
    // FilterLogging_StringList.Add('OutObj: ' + OutObj);

    // -------------------------------------------------------------------
    // Save the manual log file
    // -------------------------------------------------------------------
    FilterLogging_StringList.SaveToFile(GetCurrentCaseDir + 'Logs' + '\' + UpperCase(ExtractFileName(CmdLine.Path)));
    CreatePDF(FilterLogging_StringList, 'Keywords Used.pdf', 8);

  finally
    aKeywords_StringList.free;
    FilterLogging_StringList.free;
  end;
end;

// ------------------------------------------------------------------------------
// Run the filter using the OutObject
// ------------------------------------------------------------------------------
function FilterEntry(anEntry: TEntry; InObj: string): integer;
begin
  Result := -1; // Exclude
  if assigned(anEntry) then
  begin
    // if RegexMatch(anEntry.EntryName, InObj, False) then
    if RegexMatch(anEntry.FullPathName, InObj, False) then
    begin
      Result := 1; // Include
    end;
  end;
end;

begin

end.
